/*************************************************************************
 * The contents of this file are subject to the MYRICOM MYRINET          *
 * EXPRESS (MX) NETWORKING SOFTWARE AND DOCUMENTATION LICENSE (the       *
 * "License"); User may not use this file except in compliance with the  *
 * License.  The full text of the License can found in LICENSE.TXT       *
 *                                                                       *
 * Software distributed under the License is distributed on an "AS IS"   *
 * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied.  See  *
 * the License for the specific language governing rights and            *
 * limitations under the License.                                        *
 *                                                                       *
 * Copyright 2003 - 2004 by Myricom, Inc.  All rights reserved.          *
 *************************************************************************/

#ifndef MX__RDMA_ALLOC_H
#define MX__RDMA_ALLOC_H

#define MX_RDMA_PAGE_SHIFT     MX_MCP_VPAGE_SHIFT
#define MX_RDMA_PAGE_SIZE      ((uintptr_t)1 << MX_RDMA_PAGE_SHIFT)
#define MX_RDMA_PAGE_OFFSET(x) ((x)&(MX_RDMA_PAGE_SIZE-1))
#define MX_RDMA_PAGE_ALIGN(x)  (((x)+MX_RDMA_PAGE_SIZE-1)&(~(MX_RDMA_PAGE_SIZE-1)))

/* quick and dirty allocator for the rdma window, based on a bitmap */

typedef struct {
  uint32_t *bitmap;
  int last;
  int max;
  int count;
  int send_count;
} mx_rdma_pool_t;


static inline void mx_rdma_use(mx_rdma_pool_t *pool,int num)
{
  pool->bitmap[num/32] |= (1<<(num %32));
  pool->count += 1;
}

static inline void mx_rdma_unuse(mx_rdma_pool_t *pool,int num, int send)
{
  pool->bitmap[num/32] &= ~(1<<(num %32));
  pool->count -= 1;
  pool->send_count -= send ? 1 : 0;
}


static inline int mx_rdma_is_free(mx_rdma_pool_t *pool,int num)
{
  return !(pool->bitmap[num/32] & (1<<(num %32)));
}


static inline int mx_rdma_allocate_slot(mx_rdma_pool_t *pool, int send)
{
  int start, i;
  if (pool->count >= pool->max) {
    return -1;
  }
  start = i = pool->last;
  while (!mx_rdma_is_free(pool,i)) {
    i += 1;
    if (i == pool->max)
      i = 0;
    mx_assert (i != start);
  }
  pool->last = (i + 1 >= pool->max) ? 0 : i + 1;
  mx_rdma_use(pool, i);
  mx_assert(send == 0 || send == 1);
  pool->send_count += send;
  return i;
}

static inline mx_return_t mx_rdma_init(mx_rdma_pool_t *pool, int nbentries)
{
  pool->bitmap = mx_calloc((nbentries + 31)/32, 4);
  if (!pool->bitmap)
    return MX_NO_RESOURCES;
  pool->max = nbentries;
  pool->count = 0;
  pool->send_count = 0;
  /* first alloc at one rather than zero helps debugging */
  pool->last = 1;
  mx_always_assert(pool->bitmap);
  pool->max = nbentries;
  return MX_SUCCESS;
}

static inline void mx_rdma_finalize(mx_rdma_pool_t *pool)
{
  mx_free(pool->bitmap);
  memset(pool,0,sizeof(*pool));
}

static inline void mx_rdma_free(mx_rdma_pool_t *pool)
{
  mx_free(pool->bitmap);
  pool->bitmap = 0;
}

#endif /* RDMA_ALLOC_H */
